#!/usr/bin/env python3
"""
Example: Video Recording with Cooperative Command Game

Demonstrates the new video recording capabilities in the gym wrapper.
"""

import numpy as np
import time
from gym_wrapper import CoopCommandGymEnv

def run_video_recording_example():
    """Run examples of different video recording modes."""
    
    print("🎥 Cooperative Command Game - Video Recording Examples")
    print("=" * 60)
    
    # Example 1: Individual mode (traditional - images + audio separately)
    print("\n📸 Example 1: Individual Mode (traditional)")
    print("-" * 40)
    env_individual = CoopCommandGymEnv(
        difficulty="normal",
        max_rounds=10,
        enable_audio=True,
        enable_visual=True,
        recording_mode="individual",
        seed_index=0
    )
    
    obs, info = env_individual.reset()
    print(f"Observation keys: {list(obs.keys()) if isinstance(obs, dict) else 'vector only'}")
    print(f"Score: {info['score_normalized']}")
    
    # Run a few steps
    for step in range(3):
        action = [0, 0, 50, 50]  # Simple move command
        obs, reward, terminated, truncated, info = env_individual.step(action)
        print(f"Step {step+1}: Reward={reward:.2f}, Score={info['score_normalized']}")
        if terminated:
            break
    
    env_individual.close()
    print("✅ Individual mode example completed\n")
    
    # Example 2: Video mode (frames + audio integrated into video files)
    print("\n🎬 Example 2: Video Mode (integrated video files)")
    print("-" * 40)
    env_video = CoopCommandGymEnv(
        difficulty="normal",
        max_rounds=15,
        enable_audio=True,
        enable_visual=True,
        recording_mode="video",
        video_fps=10,  # Lower FPS for demo
        recordings_dir="recordings_demo",
        seed_index=1
    )
    
    obs, info = env_video.reset(options={'session_name': 'demo_video_session'})
    print(f"Observation keys: {list(obs.keys()) if isinstance(obs, dict) else 'vector only'}")
    print(f"Video status: {env_video.get_video_status()}")
    
    # Run more steps to create meaningful video content
    for step in range(8):
        # Vary actions for more interesting video
        if step % 3 == 0:
            action = [0, 0, np.random.randint(20, 80), np.random.randint(20, 80)]  # Move
        elif step % 3 == 1:
            action = [0, 1, np.random.randint(30, 70), np.random.randint(30, 70)]  # Attack
        else:
            action = [0, 4, 0, 0]  # Status report
        
        obs, reward, terminated, truncated, info = env_video.step(action)
        
        # Check video data in observation
        if isinstance(obs, dict) and 'video_data' in obs:
            import json
            video_data = json.loads(obs['video_data'])
            print(f"Step {step+1}: Frames={video_data.get('frames', 0)}, Duration={video_data.get('duration_seconds', 0):.1f}s")
        
        if terminated:
            break
        
        time.sleep(0.1)  # Small delay for more realistic timing
    
    # Save intermediate video without stopping recording
    success, message = env_video.save_current_session_video("recordings_demo/intermediate_save.mp4")
    if success:
        print(f"✅ Intermediate save: {message}")
    else:
        print(f"⚠️ Intermediate save failed: {message}")
    
    env_video.close()
    print("✅ Video mode example completed\n")
    
    # Example 3: Both mode (individual + video)
    print("\n🎭 Example 3: Both Mode (individual + video)")
    print("-" * 40)
    env_both = CoopCommandGymEnv(
        difficulty="medium",
        max_rounds=12,
        enable_audio=True,
        enable_visual=True,
        recording_mode="both",
        video_fps=15,
        recordings_dir="recordings_demo",
        seed_index=2
    )
    
    obs, info = env_both.reset(options={'session_name': 'demo_both_session'})
    print(f"Observation keys: {list(obs.keys()) if isinstance(obs, dict) else 'vector only'}")
    
    # Demonstrate all observation types
    if isinstance(obs, dict):
        print(f"Vector shape: {obs['vector'].shape}")
        if 'image' in obs:
            print(f"Image shape: {obs['image'].shape}")
        if 'audio' in obs:
            import json
            audio_data = json.loads(obs['audio'])
            print(f"Audio events: {len(audio_data)}")
        if 'video_data' in obs:
            import json
            video_data = json.loads(obs['video_data'])
            print(f"Video recording: {video_data.get('frames', 0)} frames")
    
    # Run some steps with team commands
    commands = [
        [4, 4, 0, 0],  # All team status
        [0, 2, 40, 60],  # Individual defend
        [1, 3, 70, 30],  # Individual recon
        [4, 0, 25, 75],  # All team move
        [0, 1, 50, 50],  # Individual attack
    ]
    
    for step, action in enumerate(commands):
        obs, reward, terminated, truncated, info = env_both.step(action)
        print(f"Step {step+1}: Command={action}, Reward={reward:.2f}")
        
        if terminated:
            break
        
        time.sleep(0.1)
    
    env_both.close()
    print("✅ Both mode example completed\n")
    
    print("🎯 All video recording examples completed!")
    print("📁 Check 'recordings_demo/' directory for generated video files")
    print("\nVideo file outputs:")
    print("- .mp4 files: Actual video recordings")
    print("- .json files: Audio events and metadata")
    print("- Videos may include integrated audio if audio files were found")

def run_manual_recording_control():
    """Demonstrate manual video recording control."""
    print("\n🎛️ Manual Recording Control Example")
    print("-" * 40)
    
    # Create environment without auto-recording
    env = CoopCommandGymEnv(
        difficulty="normal",
        max_rounds=20,
        enable_audio=True,
        enable_visual=True,
        recording_mode="individual",  # Start in individual mode
        seed_index=3
    )
    
    obs, info = env.reset()
    print("✅ Environment initialized (no auto-recording)")
    
    # Manually start video recording
    success = env.start_video_recording("manual_control_session")
    if success:
        print("🎥 Manual video recording started")
    else:
        print("⚠️ Failed to start manual recording")
        return
    
    # Run some steps
    for step in range(5):
        action = [step % env.num_members, step % 5, 30 + step*10, 40 + step*5]
        obs, reward, terminated, truncated, info = env.step(action)
        
        # Check recording status
        status = env.get_video_status()
        print(f"Step {step+1}: Recording={status['recording']}, Frames={status.get('frames', 0)}")
        
        if terminated:
            break
    
    # Manually stop and save
    success, message = env.stop_video_recording("recordings_demo/manual_session.mp4")
    if success:
        print(f"✅ Manual recording saved: {message}")
    else:
        print(f"⚠️ Manual recording save failed: {message}")
    
    env.close()
    print("✅ Manual recording control example completed")

if __name__ == "__main__":
    # Run the examples
    try:
        run_video_recording_example()
        run_manual_recording_control()
    except KeyboardInterrupt:
        print("\n⏹️ Examples interrupted by user")
    except Exception as e:
        print(f"\n❌ Error running examples: {e}")
        import traceback
        traceback.print_exc() 